
#ifndef NEOTRINO_SCREEN_GRAB_PROTOCOL_H
#define NEOTRINO_SCREEN_GRAB_PROTOCOL_H

#include <vector>
#include <string>
#ifdef SDK_SIMPLE_INCLUDE
#include "ScreenRecorderExport.h"
#else
#include <ScreenRecorder/ScreenRecorderExport.h>
#endif // SDK_SIMPLE_INCLUDE

/*! Screen grab interface. */
class DLL_DECL_SR ScreenGrabProtocol {
public:
    /**
     * Get Windows ids and titles.
     * @param window_ids Window ids.
     *                   On Windows: HWND cast to intptr_t.
     *                   On OSX: integer window number.
     * @param titles Window titles, use utf-8 encoding.
     */
    static bool GetWindows(std::vector<intptr_t>& window_ids, std::vector<std::string>& titles);
    /**
     * Get screen ids and names.
     * @param screen_ids Screen ids.
     *                   On Windows: integer display device index.
     *                   On OSX: CGDirectDisplayID cast to intptr_t.
     * @param names Screen names, use utf-8 encoding.
     *              On Windows: Same as member 'DeviceName' of structure 'DISPLAY_DEVICE'.
     *              On OSX: Same as id.
     */
    static bool GetScreens(std::vector<intptr_t>& screen_ids, std::vector<std::string>& names);

    /**
    * Grab window to bitmap.
      @param window_id Windows. On Windows: HWND cast to intptr_t, On OSX: integer window number.
      @return On Windows: HBITMAP, On OSX: CGImageRef.
    */
    static void* GrabWindow(intptr_t window_id);

    /**
    * Grab screen to bitmap.
      @param screen_id Screen id. On Windows: integer display device index,  On OSX: CGDirectDisplayID cast to intptr_t.
      @return On Windows: HBITMAP, On OSX: CGImageRef.
    */
    static void* GrabScreen(intptr_t screen_id);

    /**
    * Grab window to file.
      @param path Utf8 coded path to save image to file.
      @param window_id Windows. On Windows: HWND cast to intptr_t, On OSX: integer window number.
    */
    static bool GrabWindow(const std::string& path, intptr_t window_id);

    /**
    * Grab screen to file.
    * @param path Utf8 coded path to save image to file.
      @param screen_id Screen id. On Windows: integer display device index,  On OSX: CGDirectDisplayID cast to intptr_t.
    */
    static bool GrabScreen(const std::string& path, intptr_t screen_id);

    ScreenGrabProtocol() = default;
    virtual ~ScreenGrabProtocol() = default;

    /**
     * Set exclude windows.
     * @param windows Windows. See also GetWindows.
     * @param alphas alpha values.
     */
    virtual bool SetExcludeWindows(const std::vector<intptr_t>& windows, const std::vector<bool>& alphas = std::vector<bool>()) { return false; }

    /**
     * Add exclude window.
     * @param window_id window id. See also GetWindows.
     * @param alpha alpha value.
     */
    virtual bool AddExcludeWindow(intptr_t window_id, bool alpha = false) { return false; }

    /**
     * Remove exclude window.
     * @param window_id window id. See also GetWindows.
     */
    virtual bool RemoveExcludeWindow(intptr_t window_id) { return false; }

    /**
     * Clear exclude window.
     */
    virtual bool ClearExcludeWindows() { return false; }

    /**
     * Begin full screen exclude. Only valid for screen share.
     * @param window_id window id. See also GetWindows.
     */
    virtual void BeginFullScreenExclude(intptr_t window_id) {}

    /**
     * End full screen exclude. Only valid for screen share.
     * @param window_id window id. See also GetWindows.
     */
    virtual void EndFullScreenExclude(intptr_t window_id) {}

    /**
     * Begin minimize windows. Only valid for window share or record.
     * @param window_id window id. See also GetWindows.
     */
    virtual void BeginMinimizeWindow(intptr_t window_id) {}

    /**
     * End minimize windows. Only valid for window share or record.
     * @param window_id window id. See also GetWindows.
     */
    virtual void EndMinimizeWindow(intptr_t window_id) {}

    /**
     * Select window to grab.
     * @param window_id Windows. See also GetWindows.
     */
    virtual bool SelectWindow(intptr_t window_id) = 0;
    /**
     * Select screen to grab.
     * @param screen_id Screen id. See also GetScreens.
     */
    virtual bool SelectScreen(intptr_t screen_id) = 0;
};

#endif // NEOTRINO_SCREEN_GRAB_PROTOCOL_H
